/* Copyright (C) 2013-2018 RealVNC Ltd.  All Rights Reserved.
 */

#ifndef VNCDECODER_H_013E722A_AB5D_4BA2_962E_83202E7C27AB
#define VNCDECODER_H_013E722A_AB5D_4BA2_962E_83202E7C27AB

#include "vnctypes.h"

/**
 * \file vncdecoder.h
 *
 * This file defines types for use with pluggable decoders provided to
 * the VNC Viewer SDK.
 *
 * \section decoder_pixel_flow Pixel Data Flow
 *
 *
 * If no pluggable decoder is present for a particular format then the pixel
 * data path for the Viewer SDK can be seen in the following diagram:
 *
 * \dot
 * digraph G {
 *     node [style=filled, color=lightgrey, shape=box]
 *     rankdir = LR
 *
 *     Server [label="Server Device", shape=box, color=grey, fontcolor=white, style = filled];
 *     Display [label="Display Output"];
 *
 *     subgraph cluster_viewersdk {
 *         label = "Viewer SDK"
 *         color = grey
 *         RFBDecoder [label="RFB Protocol Decoder"];
 *         PixelDecoder [label="Pixel Data Decoder"];
 *         PixelConverter [label="Pixel Format Converter"];
 *     }
 *
 *     subgraph cluster_viewer {
 *         label = "Viewer"
 *         color = grey
 *         FBImage [label="Framebuffer Image"];
 *     }
 *
 *     subgraph cluster_gpu {
 *         label = "GPU"
 *         color = grey
 *         Display [label="Display Output"];
 *     }
 *
 *     Server -> RFBDecoder [style=dotted];
 *     RFBDecoder -> PixelDecoder;
 *     PixelDecoder -> PixelConverter;
 *     PixelConverter ->FBImage [constraint=false];
 *     FBImage -> Display;
 * }
 * \enddot
 *
 * Once registered, a pluggable decoder will be provided with the
 * encoded pixel data that the Viewer SDK receives.  The pluggable
 * decoder can then provide the raw pixel data back to the SDK by
 * calling VNCDecoderDrawRectangle().
 *
 * When pixel data is received in an encoding supported by a loaded
 * pluggable decoder, the pixel path is as follows:
 *
 * \dot
 * digraph G {
 *     node [style=filled, color=lightgrey, shape=box]
 *     rankdir = LR
 *
 *     Server [label="Server Device", shape=box, color=grey, fontcolor=white, style = filled];
 *     Display [label="Display Output"];
 *
 *     subgraph cluster_viewersdk {
 *         label = "Viewer SDK"
 *         color = grey
 *         RFBDecoder [label="RFB Protocol Decoder"];
 *         PixelConverter [label="Pixel Format Converter"];
 *     }
 *
 *     subgraph cluster_viewer {
 *         label = "Viewer"
 *         color = grey
 *         FBImage [label="Framebuffer Image"];
 *     }
 *
 *     subgraph cluster_decoder {
 *         label = "Pluggable Decoder"
 *         color = grey
 *         PixelDecoder [label="Pixel Data Decoder"];
 *     }
 *
 *     subgraph cluster_gpu {
 *         label = "GPU"
 *         color = grey
 *         Display [label="Display Output"];
 *     }
 *
 *     Server -> RFBDecoder [style=dotted];
 *     RFBDecoder -> PixelDecoder;
 *     PixelDecoder -> PixelConverter;
 *     PixelConverter -> FBImage [constraint=false];
 *     FBImage -> Display;
 * }
 * \enddot
 *
 * Alternatively, a pluggable decoder is able to provide the data to
 * a pluggable renderer, in an arbitrary format known in advance by
 * both the decoder and the renderer.
 *
 * To do this, the decoder should call VNCDecoderDrawCustomData() rather
 * than VNCDecoderDrawRectangle().
 *
 * As before, the SDK will continue to decode data in encodings for
 * which there is no suitable pluggable decoder present. However,
 * when a pluggable renderer is present, this data is passed to the
 * renderer, rather than the viewer's framebuffer.
 *
 * When pixel data is received in an encoding supported by a loaded
 * pluggable decoder, the pixel path is as follows:
 *
 * \dot
 * digraph G {
 *     node [style=filled, color=lightgrey, shape=box]
 *     rankdir = LR
 *
 *     Server [label="Server Device", shape=box, color=grey, fontcolor=white, style = filled];
 *     Display [label="Display Output"];
 *
 *     subgraph cluster_viewersdk {
 *         label = "Viewer SDK"
 *         color = grey
 *         RFBDecoder [label="RFB Protocol Decoder"];
 *     }
 *
 *     subgraph cluster_viewer {
 *         label = "Viewer"
 *         color = grey
 *         ViewerUI [label="Viewer User Interface"];
 *     }
 *
 *     subgraph cluster_decoder {
 *         label = "Pluggable Decoder"
 *         color = grey
 *         PixelDecoder [label="Pixel Data Decoder"];
 *     }
 *
 *     subgraph cluster_renderer {
 *         label = "Pluggable Renderer"
 *         color = grey
 *         PixelConverter [label="Pixel Format Converter"];
 *         FBSurface [label="Framebuffer Surface"];
 *         Display [label="Display Output"];
 *     }
 *
 *     Server -> RFBDecoder [style=dotted];
 *     RFBDecoder -> PixelDecoder;
 *     PixelDecoder -> PixelConverter [constraint=false];
 *     PixelConverter ->FBSurface;
 *     FBSurface -> Display;
 *     ViewerUI -> Display;
 * }
 * \enddot
 *
 * And when pixel data is received in an encoding which is not supported
 * by any loaded pluggable decoder, the data is decoded by the SDK before
 * being passed to the renderer:
 *
 * \dot
 * digraph G {
 *     node [style=filled, color=lightgrey, shape=box]
 *     rankdir = LR
 *
 *     Server [label="Server Device", shape=box, color=grey, fontcolor=white, style = filled];
 *     Display [label="Display Output"];
 *
 *     subgraph cluster_viewersdk {
 *         label = "Viewer SDK"
 *         color = grey
 *         RFBDecoder [label="RFB Protocol Decoder"];
 *         PixelDecoder [label="Pixel Data Decoder"];
 *     }
 *
 *     subgraph cluster_viewer {
 *         label = "Viewer"
 *         color = grey
 *         ViewerUI [label="Viewer User Interface"];
 *     }
 *
 *     subgraph cluster_renderer {
 *         label = "Pluggable Renderer"
 *         color = grey
 *         PixelConverter [label="Pixel Format Converter"];
 *         FBSurface [label="Framebuffer Surface"];
 *         Display [label="Display Output"];
 *     }
 *
 *     Server -> RFBDecoder [style=dotted];
 *     RFBDecoder -> PixelDecoder;
 *     PixelDecoder -> PixelConverter [constraint=false];
 *     PixelConverter ->FBSurface;
 *     FBSurface -> Display;
 *     ViewerUI -> Display;
 * }
 * \enddot
 *
 * \section decoder_state_machine Decoder State Machine
 *
 * The lifecycle of a pluggable decoder is represented by the
 * following state machine:
 *
 * \dot
 * digraph G {
 *     Start [shape=diamond]
 *     node [style=filled, color=lightgrey, shape=box]
 *     rankdir = LR
 *
 *     Start -> Unregistered [constraint=false];
 *
 *     Advertised -> Registered [label="Destroy", URL="\ref VNCDecoderDestroy"];
 *     Unregistered -> Registered [label="Register", URL="\ref VNCViewerRegisterDecoder", constraint=false];
 *     Registered -> Connected [label="Create", URL="\ref VNCDecoderCreate"];
 *     Connected -> Registered [label="Destroy", URL="\ref VNCDecoderDestroy"];
 *     Connected -> Advertised [label="Advertised", URL="\ref VNCDecoderAdvertised"];
 *     Advertised -> Connected [label="Hidden", URL="\ref VNCDecoderHidden"];
 * }
 * \enddot
 *
 * \see VNCViewerRegisterDecoder, VNCViewerLoadDecoderFromFile
 */

/**
 * \typedef VNCDecoderEventHandle
 *
 * A VNCDecoderEventHandle is a platform-specific mechanism for
 * the SDK to detect that there has been activity on a pluggable
 * decoders' context.
 *
 * See VNCDecoderContextGetEventHandles() for a full discussion of how
 * the SDK uses VNCDecoderContextEventHandle.
 *
 * \see VNCDecoderContextGetEventHandles(), VNCDecoderContextActivity()
 */
#ifdef _WIN32
    #include <windows.h>
    typedef HANDLE VNCDecoderEventHandle;
#else
    typedef int VNCDecoderEventHandle;
#endif

/**
 * An enumeration used to describe the direction of activity
 * of interest for each VNCDecoderEventHandle.
 *
 * Due to VNCDecoderEventHandle being waitable kernel objects on
 * Windows, using any value which is not VNCDecoderEventDirectionNone
 * will cause the kernel object to be waited on.
 */
typedef enum
{
  /**
   * Used to indicate that the event handle should not be monitored.
   */
  VNCDecoderEventDirectionNone  = 0x0,
  /**
   * Used to indicate that the event handle should only be monitored
   * for read activity.
   */
  VNCDecoderEventDirectionRead  = 0x1,
  /**
   * Used to indicate that the event should only be monitored for
   * write activity.
   */
  VNCDecoderEventDirectionWrite = 0x2,
  /**
   * Used to indicate that the event should
   * be monitored for both read and write activity.
   */
  VNCDecoderEventDirectionBoth  = 0x3
} VNCDecoderEventDirection;

/**
 * Represents a loaded pluggable decoder.
 *
 * This type is opaque to the SDK, which will pass the value returned
 * by VNCDecoderInitialize() to any further decoder interface calls
 * that may require it.
 */
typedef struct VNCDecoderFactoryImpl VNCDecoderFactory;

/**
 * Represents a context created using a pluggable decoder.
 *
 * This type is opaque to the SDK, which will pass the value returned by
 * VNCDecoderContextCreate() to any further decoder interface calls that may
 * require it.
 */
typedef struct VNCDecoderImpl VNCDecoder;

/**
 * Identifies the internal data structures of the SDK that correspond to a
 * loaded pluggable decoder factory.
 *
 * This type is opaque to the decoder implementation, which must pass
 * the value supplied to VNCDecoderFactoryInitialize() to any
 * supporting API calls that may require it.
 */
typedef void *VNCDecoderFactoryContext;

/**
 * Identifies the internal data structures of the SDK that correspond to a
 * loaded pluggable decoder.
 *
 * This type is opaque to the decoder implementation, which must pass the value
 * supplied to VNCDecoderCreate() to any supporting API calls that may
 * require it.
 */
typedef void *VNCDecoderContext;

/**
 * Identifies the decoder viewer context provided by the user of the viewer SDK.
 *
 * This type is opaque to the SDK, which will just pass the pointer provided to
 * the call to VNCViewerRegisterDecoder() or VNCViewerLoadDecoderFromFile().
 */
typedef void *VNCDecoderViewerContext;

/**
 * \brief Error codes that may be returned by the implementations of functions
 * in VNCDecoderInterface and VNCDecoderSupportingAPI.
 *
 * The error codes are split into two ranges:
 *
 *  - 0 to (VNCDecoderErrorVENDOR - 1) - these error codes are a mixture of
 *    error codes (e.g.  VNCDecoderErrorNoMemory) and
 *    other error conditions that are pre-defined by RealVNC-provided decoders.
 *    You are encouraged to reuse error codes in this range, rather than
 *    defining your own, as long as it makes sense to do so.
 *  - VNCDecoderErrorVENDOR and above - this range of error codes is reserved
 *    for the use of third parties developing decoders.  It is intended for
 *    error conditions that are specific to particular decoder implementations,
 *    and that do not map closely to the codes in the 0 to
 *    (VNCDecoderErrorVENDOR - 1) range.
 */
typedef enum
{
    /**
     * The operation was successful - no error occurred.
     */
    VNCDecoderErrorNone = 0,

    /**
     * No memory was available for the requested operation.
     */
    VNCDecoderErrorNoMemory = 1,

    /**
     * The required resources are unavailable.
     */
    VNCDecoderErrorResourceUnavailable = 2,

    /**
     * The decoder requires a licensed feature for which the SDK does not have a
     * license.
     */
    VNCDecoderErrorFeatureNotLicensed = 3,

    /*
     * The value of one of the parameters to the call is not valid.
     */
    VNCDecoderErrorInvalidParameter = 4,

    /*
     * The SDK encountered an unknown internal error.
     */
    VNCDecoderErrorSDKInternal = 5,

    /*
     * There was a problem with the incoming data.
     */
    VNCDecoderErrorMalformedData = 6,

    /*
     * The required resources encountered an error.
     */
    VNCDecoderErrorResourceError = 7,

    /*
     * The decoder encountered an unknown internal error.
     */
    VNCDecoderErrorInternal = 8,

    /**
     * Start of range of third-party decoder-specific error codes.
     */
    VNCDecoderErrorVENDOR = 0x10000
} VNCDecoderError;

/**
 * A collection of bitwise flags for indicating
 * the functionality supported by a particular decoder.
 */
typedef enum {
  /**
   * If set then the decoder can make use of displaying
   * pixel data via an overlay.
   *
   * If VNCDecoderSupportFlagPixelData is also set then
   * the decoder can choose to use an overlay or
   * calls to VNCDecoderDrawRectangle().
   *
   * If a decoder does set both this flag and
   * VNCDecoderSupportFlagPixelData then it is the responsibility of the
   * decoder and SDK to ensure that the framebuffer data provided to
   * the SDK is correctly blended with the data displayed via the
   * overlay.
   */
  VNCDecoderSupportFlagOverlay = 0x1,
  /**
   * If set then the decoder can make use of returning pixel data to
   * the SDK via calls to VNCDecoderDrawRectangle().
   *
   * If VNCDecoderSupportFlagOverlay is also set then the
   * decoder can choose to use an instead overlay.
   *
   * If a decoder does set both this flag and
   * VNCDecoderSupportFlagOverlay then it is the responsibility of the
   * decoder and SDK to ensure that the framebuffer data provided to
   * the SDK is correctly blended with the data displayed via the
   * overlay.
   *
   */
  VNCDecoderSupportFlagPixelData = 0x2,
  /**
   * Equivalent to setting both VNCDecoderSupportFlagOverlay and
   * VNCDecoderSupportFlagPixelData.
   */
  VNCDecoderSupportFlagOverlayAndPixelData = 0x3,
} VNCDecoderSupportFlag;

/**
 * A collection of encoding numbers, as allocated by The Internet Assigned
 * Numbers Authority (IANA). One of these values should be used in the
 * encodingNumber field of the VNCDecoderSupportInfo structure.
 *
 * For further details see:
 *   http://www.iana.org/assignments/rfb/rfb.xhtml#rfb-4
 */
typedef enum {
    VNCDecoderTypeRaw = 0,
    VNCDecoderTypeTRLE = 15,
    VNCDecoderTypeZRLE = 16,
    VNCDecoderTypeJPEG = 21,
    VNCDecoderTypeH264 = 20,
    VNCDecoderTypeVAH264 = 0x48323634,
    VNCDecoderTypeSLRLE = -525,
} VNCDecoderType;

/**
 * \brief A structure describing the supported features of a pluggable decoder.
 */
typedef struct
{
  /**
   * The encoding number that this decoder supports.
   *
   * This should be one of the allocated framebuffer encoding numbers
   * which are managed by The Internet Assigned Numbers Authority (IANA).
   *
   * For further details see:
   *   http://www.iana.org/assignments/rfb/rfb.xhtml#rfb-4
   */
  VNCDecoderType encodingNumber;
  /**
   * The human readable name for the encoding, encoded as modified
   * UTF-8 with a NUL terminator.
   *
   * This can be NULL for encodings built into the SDK.
   *
   * This should have been allocated using VNCDecoderFactoryAlloc().
   */
  char *pEncodingName;
  /**
   * The human readable name for this implementation of the decoder,
   * encoded as modified UTF-8 with a NUL terminator.
   *
   * This might include the company name or details of the specific
   * hardware requirements. This will only be used by the SDK in
   * logging and debugging messages.
   *
   * This should have been allocated using VNCDecoderFactoryAlloc().
   */
  char *pDecoderName;
  /**
   * The human readable version for this implementation of the decoder,
   * encoded as modified UTF-8 with a NUL terminator.
   *
   * The format of this string is entirely up to the decoder, it will
   * only be used by the SDK in logging and debugging messages.
   *
   * This should have been allocated using VNCDecoderFactoryAlloc().
   */
  char *pDecoderVersion;
  /**
   * A collection of bitwise OR'd flags describing the supported
   * functionality for this decoder.
   *
   * Any unknown flags should be set to zero.
   */
  VNCDecoderSupportFlag flags;
} VNCDecoderSupportInfo;

/**
 * \brief A structure describing the connection responsible for
 * decoder creation.
 *
 * This structure is provided by the Viewer SDK to allow a decoder
 * to be aware of various per connection parameters.
 *
 * \see VNCDecoderCreate
 */
typedef struct
{
  /**
   * The major version of the RFB protocol in use.
   *
   * This is required as some encodings have different data formats
   * between RFB versions.
   */
  vnc_uint16_t rfbVersionMajor;
  /**
   * The minor version of the RFB protocol in use.
   *
   * This is required as some encodings have different data formats
   * between RFB versions.
   */
  vnc_uint16_t rfbVersionMinor;
  /**
   * The pixel format being used for the SDK copy of the remote frame
   * buffer.
   *
   * This is different from the pixel format in use over the
   * connection, which is provided via the
   * VNCDecoderPixelFormatChange() call.
   */
  VNCPixelFormat framebufferPixelFormat;
  /**
   * The native pixel format for the server.
   *
   * This will be the initial pixel format used over the wire until a
   * call to VNCDecoderPixelFormatChange() occurs.
   */
  VNCPixelFormat serverNativePixelFormat;
} VNCDecoderCreationInfo;

/**
 * \brief A structure containing the name and version of the currently loaded
 * pluggable renderer.
 */
typedef struct {

  /**
   * The human readable name of the renderer, encoded as modified UTF-8 with
   * a NUL terminator. This corresponds to the value in VNCRendererSupportInfo
   * with the same name.
   *
   * Ownership of this string is transferred to the decoder when it is provided
   * with this struct. It should be freed using VNCDecoderFactoryFree when it
   * is no longer needed.
   */
   char *pRendererName;

  /**
   * The human readable version of the renderer, encoded as modified UTF-8 with
   * a NUL terminator. This corresponds to the value in VNCRendererSupportInfo
   * with the same name.
   *
   * Ownership of this string is transferred to the decoder when it is provided
   * with this struct. It should be freed using VNCDecoderFactoryFree when it
   * is no longer needed.
   */
   char *pRendererVersion;

} VNCRendererIdentificationInfo;

/**
 * Identifies the internal data structures of the SDK that correspond to a
 * non-blocking input stream.
 *
 * This type is opaque to the decoder implementation, which must pass the value
 * supplied to VNCDecoderRectangleData() to any supporting API calls that may
 * require it (namely, VNCDecoderNonBlockingInputStreamRead).
 */
typedef void *VNCDecoderNonBlockingInputStream;


/**
 * \brief Called by the SDK prior to unloading a decoder DLL or shared object.
 *
 * The implementation should free all resources associated with the
 * decoder implementation.
 *
 * The SDK will call VNCDecoderDestroy() for all VNCDecoder objects created by
 * this decoder before calling this method.
 *
 * \param pDecoderFactory The decoder factory to be unloaded.
 */
typedef void VNCCALL
VNCDecoderFactoryTerminate(VNCDecoderFactory *pDecoderFactory);

/**
 * \brief Called by the SDK to request that the decoder factory fills in
 * a structure with support information.
 *
 * Ownership of any strings contained in pDecoderSupportInfo is transferred
 * to the SDK if VNCDecoderErrorNone is returned.
 *
 * \param pDecoderFactory The decoder factory to request information about.
 * \param pDecoderSupportInfo On success this should have been filled
 * with information describing the decoder.
 * \param decoderSupportInfoSize The size of the memory pointed to by
 * pDecoderSupportInfo.
 *
 * \return VNCDecoderErrorNone on success, or some other VNCDecoderError on
 * failure.
 */
typedef VNCDecoderError VNCCALL
VNCDecoderFactoryGetSupportInfo(VNCDecoderFactory *pDecoderFactory,
                                VNCDecoderSupportInfo *pDecoderSupportInfo,
                                size_t decoderSupportInfoSize);

/**
 * \brief Called by the SDK at some point during or after the start of
 * a connection to create a per-connection instance of a decoder.
 *
 * Failure to create a decoder will not be treated by the SDK as a
 * fatal error. If a built-in decoder is available then this will be
 * used instead, otherwise no support for the encoding will be
 * advertised to the server.
 *
 * The provided VNCDecoderCreationInfo is only valid until the call to
 * this method returns. A decoder should take a copy of this structure
 * if it may require it at a later time.
 *
 * \param pDecoderFactory The decoder factory to use.
 * \param decoderContext Opaque data type to be provided to applicable
 * supporting APIs.
 * \param pDecoderCreationInfo Information about the connection creating
 * the the decoder.
 * \param decoderCreationInfoSize Size of the memory pointed to by
 * pDecoderCreationInfo.
 *
 * \return A pointer for use in other decoder calls by the SDK, or
 * NULL on failure.
 *
 * \see VNCDecoderDestroy
 */
typedef VNCDecoder *VNCCALL
VNCDecoderCreate(VNCDecoderFactory *pDecoderFactory,
                 VNCDecoderContext decoderContext,
                 const VNCDecoderCreationInfo *pDecoderCreationInfo,
                 size_t decoderCreationInfoSize);

/**
 * \brief Called by the SDK to destroy a previously created decoder.
 *
 * This will be called by the SDK as part of a connection being
 * destroyed. The pointer provided will match one returned from a
 * previous call to VNCDecoderCreate.
 *
 * \param pDecoder The decoder to destroy.
 */
typedef void VNCCALL
VNCDecoderDestroy(VNCDecoder *pDecoder);

/**
 * \brief Called by the SDK to indicate that the decoder may be used
 * soon.
 *
 * The SDK will call this method after it has told the server that the
 * encoding handled by this decoder is available for use in the
 * connection.
 *
 * \param pDecoder The decoder which has been advertised.
 * \return VNCDecoderErrorNone on success, or some other VNCDecoderError on
 * failure.
 */
typedef VNCDecoderError VNCCALL
VNCDecoderAdvertised(VNCDecoder *pDecoder);

/**
 * \brief Called by the SDK to indicate that the decoder may not be used
 * soon.
 *
 * The SDK will call this method after it has told the server that the
 * encoding handled by this decoder is no longer available for use in
 * the connection.
 *
 * Some servers will not respect such a message, so the decoder
 * should still be prepared to handle rectangles, even if doing so
 * requires allocating some resources on demand.
 *
 * \param pDecoder The decoder which has been hidden.
 * \return VNCDecoderErrorNone on success, or some other VNCDecoderError on
 * failure.
 */
typedef VNCDecoderError VNCCALL
VNCDecoderHidden(VNCDecoder *pDecoder);

/**
 * \brief Called by the SDK to retrieve the current
 * VNCDecoderEventHandle objects for the given decoder.
 *
 * Each decoder can provide zero of more handles for the SDK to
 * monitor for it. The SDK obtains these event handles by calling this
 * method on every iteration of its main loop. When activity is detected
 * on these handles the SDK will call VNCDecoderActivity(). This prevents
 * the decoder from needing to run a separate thread for performing such
 * asynchronous actions.
 *
 * On Windows, VNCDecoderEventHandle is a HANDLE that must identify
 * a waitable kernel object. On UNIX platforms,
 * VNCDecoderEventHandle is a file descriptor suitable for use in a
 * select() call.
 *
 * On successful return, *ppEventHandles should be set to point to an
 * array of handles and *ppEventDirections should point to an equal
 * sized array of VNCDecoderEventDirection values. These arrays should
 * remain valid until the next call to VNCDecoderGetEventHandles() or
 * VNCDecoderDestory() for the decoder.
 *
 * \param pDecoder The decoder to use.
 * \param ppEventHandles An out parameter pointing to an array of event handles.
 * \param ppEventDirection An out parameter pointing to an array indicating the
 * direction of interest for the handles in ppEventHandles.
 * \param eventHandleCount The number of event handles in the arrays pointed to
 * by *ppEventHandles and *ppEventDirection.
 * \return VNCDecoderErrorNone on success, or some other VNCDecoderError on
 * failure.
 */
typedef VNCDecoderError VNCCALL
VNCDecoderGetEventHandles(VNCDecoder *pDecoder,
                          VNCDecoderEventHandle **ppEventHandles,
                          VNCDecoderEventDirection **ppEventDirections,
                          size_t *eventHandleCount);

/**
 * \brief Called by the SDK when activity is detected on an event handle.
 *
 * When activity is detected in the appropriate direction on one or
 * more of the event handles provided by the previous call to
 * VNCDecoderGetEventHandles() then the SDK will call this method.
 *
 * The decoder should take care to clear any pending activity on the
 * handles or to not include them in the next call to
 * VNCDecoderGetEventHandles(), otherwise the SDK will call this method
 * almost immediately again.
 *
 * \param pDecoder The decoder to use.
 * \return VNCDecoderErrorNone on success, or some other VNCDecoderError on
 * failure.
 */
typedef VNCDecoderError VNCCALL
VNCDecoderActivity(VNCDecoder *pDecoder);

/**
 * \brief Called by the SDK to indicate that the pixel format for data
 * on the wire has changed.
 *
 * The pixel format in use over the RFB protocol can change during the course
 * of a connection. This method is called by the SDK so that it can take any
 * changes in pixel format into account when decoding encoded data.
 *
 * pNewPixelFormat is only valid for the duration of this method
 * call, so it is recommended that the decoder makes a copy of it
 * before returning.
 *
 * \param pDecoder The decoder to use.
 * \param pNewPixelFormat The new pixel format to use.
 * \return VNCDecoderErrorNone on success, or some other VNCDecoderError on
 * failure.
 */
typedef VNCDecoderError VNCCALL
VNCDecoderPixelFormatChange(VNCDecoder *pDecoder,
                            const VNCPixelFormat *pNewPixelFormat);

/**
 * \brief Called by the SDK when the decoder is first created, to
 * notify it of the current remote display size. It is called again
 * whenever the display size changes.
 *
 * Any operations needed to accomodate the new remote display size should
 * be performed in response to receiving this method call.
 *
 * \param pDecoder The decoder to use.
 * \param width The new width of the server display.
 * \param height The new height of the server display.
 * \return VNCDecoderErrorNone on success, or some other VNCDecoderError on
 * failure.
 */
typedef VNCDecoderError VNCCALL
VNCDecoderDesktopResize(VNCDecoder *pDecoder,
                        vnc_uint16_t width,
                        vnc_uint16_t height);

/**
 * \brief Called by the SDK to indicate the start of a rectangle of
 * encoded pixel data.
 *
 * Rejecting decoding of a rectangle will cause the connection to be
 * terminated. While the SDK might contain an internal decoder could
 * handle the data, most encodings have complex state which depends
 * on all the previously decoded data. Because the SDK lacks this state
 * it is not possible for it to take over handling of the decoding.
 *
 * This will always be called prior to any calls to
 * VNCDecoderRectangleData().  The end of the rectangle will be
 * signalled by a call to VNCDecoderRectangleEnd().
 *
 * \param pDecoder The decoder to use.
 * \param pRectangle The area of the remote display covered by this rectangle.
 * \return VNCDecoderErrorNone on success, or some other VNCDecoderError on
 * failure.
 *
 * \see VNCDecoderRectangleData, VNCDecoderRectangleEnd
 */
typedef VNCDecoderError VNCCALL
VNCDecoderRectangleStart(VNCDecoder *pDecoder,
                         const VNCRectangle *pRectangle);

/**
 * \brief Called by the SDK to provide rectangle data to the decoder.
 *
 * This method will be called when new rectangle data is received
 * from the server, repeating until *pEndOfData is set to true.
 *
 * The VNCDecoderNonBlockingInputStreamRead method in the supporting
 * API should be used to read data from the supplied input stream.
 *
 * The total number of bytes read from the input stream should be
 * precisely equal to the length of the rectangle. For example,
 * if the rectangle is 100 bytes long, precisely 100 bytes of data
 * must be read from the input stream before *pEndOfData is set.
 *
 * In any particular call of this method, it is advisable to
 * consume as much data as possible (without going past the end
 * of the rectangle), since the method will only be called when
 * new data arrives. Also, there is a limit to how much unconsumed
 * data may be left in the input stream's buffer, so it may be
 * advisable to process data in small chunks, or to read data into
 * a buffer managed by the decoder.
 *
 * If the decoding has finished for the current rectangle then *pEndOfData
 * should be set to true. This is especially true for RFB 3 connections
 * as otherwise the SDK will continue to pass non-rectangle data to
 * the decoder.
 *
 * A decoder should attempt to consume as much data as possible as if
 * the SDK detects a situation where the internal buffers are full and
 * no progress is being made on decoding then it will disconnect.
 *
 * \param pDecoder The decoder to use.
 * \param is The input stream providing the data to be decoded. This should
 * be passed to the VNCDecoderNonBlockingInputStreamRead method in the
 * supporting API.
 * \param pEndOfData An out parameter, if vnc_true then the decoder is
 * not interested in any more data from this rectangle.
 * \return VNCDecoderErrorNone on success, or some other VNCDecoderError on
 * failure.
 *
 * \see VNCDecoderRectangleStart, VNCDecoderRectangleEnd
 */
typedef VNCDecoderError VNCCALL
VNCDecoderRectangleData(VNCDecoder *pDecoder,
                        VNCDecoderNonBlockingInputStream is,
                        vnc_bool_t *pEndOfData);

/**
 * \brief Called by the SDK to indicate the end of a rectangle of
 * encoded pixel data.
 *
 * If any pixel data is locked then it should be unlocked
 * before returning from this method.
 *
 * \param pDecoder The decoder to use.
 * \param pRectangle The area of the remote display covered by this rectangle.
 * \return VNCDecoderErrorNone on success, or some other VNCDecoderError on
 * failure.
 *
 * \see VNCDecoderRectangleData, VNCDecoderRectangleStart
 */
typedef VNCDecoderError VNCCALL
VNCDecoderRectangleEnd(VNCDecoder *pDecoder,
                       const VNCRectangle *pRectangle);

/**
 * \brief Called by the SDK to indicate that an alpha mask has been set for
 * framebuffer pixel data.
 *
 * A decoder only needs to make use of this callback if it calls
 * VNCDecoderDrawRectangle() with a pixel format which matches the
 * local framebuffer pixel format. In this situation then SDK will
 * assume that the decoder is correctly filling in the alpha channel,
 * as requested by the an SDK user call to
 * VNCViewerSetFrameBufferAlphaMask().
 *
 * If the decoder is drawing rectangles with a different pixel format
 * to the local framebuffer then the SDK will handle the alpha channel
 * and the decoder is safe to ignore this method call.
 *
 * \param pDecoder The decoder to use.
 *
 * \param pViewer The alpha mask is specified for the framebuffer associated
 * with this viewer.
 *
 * \param alphaMax The maximum value for the alpha component is
 * \code ((2 ^ n) - 1) \endcode, where \code n \endcode is an integer
 *
 * \param alphaShift The number of right shift operations required such that
 * the least significant bit of the alpha component coincides with the least
 * significant bit of an integer type of sufficient width to accomodate the
 * pixel components. The shifts are applied after transforming the pixel
 * representation to account for the native endianess of the current platform.
 *
 * \note The alpha component of a pixel can be extracted from a pixel value as
 * \code ((pixelValue >> alphaShift) & alphaMax) \endcode
 *
 * \note The alpha mask specified here will be used to convert the pixel format
 * of any VNCCursor passed to any callback registered as a
 * VNCViewerCursorUpdateCallback into the current framebuffer pixel format.
 * Since a VNCCursor requires a pixel format with an alpha channel, the callback
 * will not be invoked unless an alpha mask is specified by calling this method.
 *
 * \return VNCDecoderErrorNone on success, or some other VNCDecoderError on
 * failure.
 */
typedef VNCDecoderError VNCCALL
VNCDecoderSetAlphaMask(VNCDecoder *pDecoder,
                       vnc_uint16_t alphaMax,
                       vnc_uint8_t alphaShift);

/**
 * \brief APIs to be implemented by the decoder and called by the SDK.
 */
typedef struct {
  /**
   * Called by the SDK prior to unloading a decoder factory DLL or
   * shared object.
   */
  VNCDecoderFactoryTerminate *vncDecoderFactoryTerminate;
  /**
   * Called by the SDK to retrieve information about the decoder
   * factory support.
   */
  VNCDecoderFactoryGetSupportInfo *vncDecoderFactoryGetSupportInfo;
  /**
   * Called by the SDK to create a decoder for a connection.
   */
  VNCDecoderCreate *vncDecoderCreate;
  /**
   * Called by the SDK to destroy a decoder for a connection.
   */
  VNCDecoderDestroy *vncDecoderDestroy;
  /**
   * Called by the SDK to inform a decoder that it has been advertised.
   */
  VNCDecoderAdvertised *vncDecoderAdvertised;
  /**
   * Called by the SDK to inform a decoder that it has been hidden.
   */
  VNCDecoderHidden *vncDecoderHidden;
  /**
   * Called by the SDK to retrieve a list of VNCDecoderEventHandle values for
   * the provided decoder.
   */
  VNCDecoderGetEventHandles *vncDecoderGetEventHandles;
  /**
   * Called by the SDK to indicate that there is activity on the event handles.
   */
  VNCDecoderActivity *vncDecoderActivity;
  /**
   * Called by the SDK to notify the decoder of a change to the pixel
   * format on the wire.
   */
  VNCDecoderPixelFormatChange *vncDecoderPixelFormatChange;
  /**
   * Called by the SDK to notify the decoder of a change to the
   * display size.
   */
  VNCDecoderDesktopResize *vncDecoderDesktopResize;
  /**
   * Called by the SDK to indicate the start of a rectangle.
   */
  VNCDecoderRectangleStart *vncDecoderRectangleStart;
  /**
   * Called by the SDK to provide rectangle data to the decoder.
   */
  VNCDecoderRectangleData *vncDecoderRectangleData;
  /**
   * Called by the SDK to indicate the end of a rectangle.
   */
  VNCDecoderRectangleEnd *vncDecoderRectangleEnd;
  /**
   * Called by the SDK to set the alpha mask for the local framebuffer.
   */
  VNCDecoderSetAlphaMask *vncDecoderSetAlphaMask;
} VNCDecoderInterface;


/**
 * \brief Called by the decoder factory to write to the SDK's log.
 *
 * VNCDecoderLog() is preferred to VNCDecoderFactoryLog() because
 * VNCDecoderLog() is guaranteed to work correctly if called from any
 * thread. VNCDecoderFactoryLog() does not work if called from a
 * thread that was started by the decoder.
 *
 * Otherwise, this method is identical to VNCDecoderLog().
 *
 * \param decoderFactoryContext The decoder VNCDecoderFactoryContext
 * that was passed to VNCDecoderFactoryInitialize().
 * \param text The text to write to the log.
 */
typedef void VNCCALL
VNCDecoderFactoryLog(VNCDecoderFactoryContext decoderFactoryContext,
                     const char *text);

/**
 * \brief Called by the decoder to allocate memory whose ownership is to be
 * transferred to the SDK.
 *
 * The start address of the returned memory is aligned to a word boundary, and
 * the memory is filled with zeroes.
 *
 * If ownership has been passed to the SDK, then the SDK will free the allocated
 * memory when it is no longer needed. If ownership is not transferred, then
 * the plugin should free the memory using VNCDecoderFactoryFree().
 *
 * \param decoderFactoryContext The VNCDecoderFactoryContext that was passed to
 * VNCDecoderFactoryInitialize().
 * \param size The size of data to be allocated.
 *
 * \return A pointer to the allocated memory or NULL on failure.
 */
typedef void *VNCCALL
VNCDecoderFactoryAlloc(VNCDecoderFactoryContext decoderFactoryContext,
                       size_t size);

/**
 * \brief Called by the decoder to allocate memory whose ownership is to be
 * transferred to the SDK. The contents of the allocated memory will be equal
 * to the supplied string.
 *
 * If ownership has been passed to the SDK, then the SDK will free the allocated
 * memory when it is no longer needed. If ownership is not transferred, then
 * the plugin should free the memory using VNCDecoderFactoryFree().
 *
 * \param decoderFactoryContext The VNCDecoderFactoryContext that was passed to
 * VNCDecoderFactoryInitialize().
 * \param str The string to copy into the allocated memory.
 *
 * \return A pointer to the allocated memory or NULL on failure.
 */
typedef char *VNCCALL
VNCDecoderFactoryAllocString(VNCDecoderFactoryContext decoderFactoryContext,
                             const char* str);

/**
 * \brief Called by the decoder to free memory which was allocated by
 * VNCDecoderFactoryAlloc or VNCDecoderFactoryAllocString, but whose
 * ownership was not transferred to the SDK.
 *
 * \param decoderFactoryContext The VNCDecoderFactoryContext that was passed to
 * VNCDecoderFactoryInitialize().
 * \param buffer The buffer to be freed.
 */
typedef void VNCCALL
VNCDecoderFactoryFree(VNCDecoderFactoryContext decoderFactoryContext,
                      void *buffer);

/**
 * \brief Called by the decoder to write to the SDK's log.
 *
 * Decoder implementations are encouraged to log significant events during
 * decoder creation and activation, and any other information that may help offline
 * diagnosis of problems.
 *
 * It is particularly important to log the reason for a failure if
 * the reason cannot easily be conveyed with an error code.
 *
 * VNCDecoderLog() is preferred to VNCDecoderFactoryLog() because
 * VNCDecoderLog() is guaranteed to work correctly if called from any
 * thread. VNCDecoderFactoryLog() does not work if called from a
 * thread that was started by the decoder.
 *
 * \param decoderContext The VNCDecoderContext that was passed to
 * VNCDecoderCreate().
 * \param text The text to write to the log.
 */
typedef void VNCCALL
VNCDecoderLog(VNCDecoderContext decoderContext, const char *text);

/**
 * \brief Called by the decoder to check whether a particular feature is
 * licensed.
 *
 * This method allows you to ensure that your custom decoders are usable only in
 * certain applications.  The feature check succeeds if the SDK is authorized
 * to use at least one of the requested features.  The features are searched
 * for both in the the licenses available to the SDK and in the set of features
 * defined by calls by the application to the SDK's API.
 *
 * If none of the features are licensed, then the decoder may either continue to
 * operate, restricting functionality as appropriate, or return
 * VNCDecoderErrorFeatureNotLicensed to the SDK, in which case the SDK will
 * terminate the session.
 *
 * \param decoderContext The VNCDecoderContext that was passed to
 * VNCDecoderCreate().
 * \param featureID The feature identifier for which to check for a license.
 * \param *pResult On successful return, *pResult is non-zero if any of the
 * features is licensed, and zero if not.
 * \retval VNCDecoderErrorNone The feature check was made successfully.  This
 * does *not* imply that any of the features is licensed; check *pResult.
 * \retval VNCDecoderErrorInvalidParameter featureIDs or pResult is NULL, or
 * featureIDCount is zero.
 *
 * pResult is non-zero if the feature is licensed, or zero if not.
 */
typedef VNCDecoderError VNCCALL
VNCDecoderLocalFeatureCheck(VNCDecoderContext decoderContext,
                            const unsigned *featureIDs,
                            size_t featureIDCount,
                            int *pResult);

/**
 * \brief Called by the decoder to draw a rectangle to the SDK's
 * framebuffer.
 *
 * \note This should only be called by the decoder between calls to
 * VNCDecoderRectangleStart() and VNCDecoderRectangleEnd().
 *
 * \param decoderContext The VNCDecoderContext that was passed to
 * VNCDecoderDecoderCreate().
 *
 * \param pPixelFormat The pixel format of the supplied data.
 *
 * \param pRectangle The area of the framebuffer to update.
 *
 * \param pBuffer The pixel data to draw to the SDK's framebuffer.
 */
typedef VNCDecoderError VNCCALL
VNCDecoderDrawRectangle(VNCDecoderContext decoderContext,
                        const VNCPixelFormat *pPixelFormat,
                        const VNCRectangle *pRectangle,
                        const vnc_uint8_t *pBuffer);

/**
 * \brief Called by the decoder to read from a VNCDecoderNonBlockingInputStream.
 *
 * Some number of bytes will be read from the supplied input stream,
 * and placed in the supplied buffer.
 *
 * This method will not block, and may return zero bytes of data.
 *
 * \param is The input stream from which to read.
 *
 * \param pBuffer The buffer into which the data should be read.
 *
 * \param maxBytes The maximum number of bytes to read.
 *
 * \return The number of bytes actually read into the buffer. Since
 * this is a non-blocking stream, this value may be zero.
 */
typedef size_t VNCCALL 
VNCDecoderNonBlockingInputStreamRead(VNCDecoderNonBlockingInputStream is,
                                     vnc_uint8_t *pBuffer,
                                     size_t maxBytes);

/**
 * \brief Called by the decoder to directly access the buffer of a
 * VNCDecoderNonBlockingInputStream.
 *
 * This method will provide a pointer to the internal buffer of an
 * input stream, so that the data can be read directly without
 * copying. The method returns the number of bytes available to
 * read at that pointer.
 *
 * Calling this method has no side effects -- the buffer position
 * remains unchanged. Two consecutive calls to this method will
 * yield the same pointer and size.
 *
 * To move forward through the buffer, use the
 * VNCDecoderNonBlockingInputStreamSkip method.
 *
 * \note Due to how the Viewer SDK handles protocol decoding, the
 * input stream's internal buffers may be segmented. It is possible
 * that, after moving forward through the buffer, more data will be
 * returned from subsequent calls to this method. Therefore, it is
 * recommended to call this method repeatedly until either the
 * decoder can't process any more data without further data, or
 * 0 bytes is returned. Failure to consume enough data may result
 * in the connection stalling.
 *
 * This method will not block, and may return 0.
 *
 * \param is The input stream to be accessed.
 *
 * \param ppBuffer An out parameter, set to point to the start of
 * the input stream's buffer. ppBuffer can't be NULL.
 *
 * \return The number of bytes available to read from the input stream.
 */
typedef size_t VNCCALL
VNCDecoderNonBlockingInputStreamPeek(VNCDecoderNonBlockingInputStream is,
                                     const vnc_uint8_t **ppBuffer);

/**
 * \brief Called by the decoder to discard a number of bytes in a
 * VNCDecoderNonBlockingInputStream.
 *
 * Calling this method will cause the input stream to skip over a
 * specified number of bytes.
 *
 * This method may cause a new buffer to be returned by
 * VNCDecoderNonBlockingInputStreamPeek, from which as much data
 * as possible should be consumed.
 *
 * \param is The input stream.
 *
 * \param bytesToSkip The number of bytes to discard. This must be
 * less than or equal to the number of available bytes returned by
 * VNCDecoderNonBlockingInputStreamPeek.
 */
typedef void VNCCALL
VNCDecoderNonBlockingInputStreamSkip(VNCDecoderNonBlockingInputStream is,
                                     size_t bytesToSkip);

/**
 * \brief Called by the decoder to retrieve the name and version
 * of the currently loaded pluggable renderer.
 *
 * This method will return false if there is no pluggable renderer
 * in use for the active connection. No changes will be made to
 * pInfoStruct.
 *
 * If a renderer is active, this method will return true, and the
 * pInfoStruct parameter will be filled, allowing the decoder to
 * determine which renderer is loaded, and whether or not the
 * renderer will understand custom data provided by this decoder.
 *
 * If the renderer is recognised by the decoder, the decoder can use the
 * vncDecoderDrawCustomData() method to provide arbitrary data directly
 * to the renderer.
 *
 * If this method returns false, or the decoder does not recognise the
 * currently loaded renderer, the decoder may choose to gracefully
 * fall back to rendering using the VNCDecoderDrawRectangle method.
 *
 * The value returned by this method, and the strings provided to
 * pInfoStruct, will remain constant over the course of a connection.
 * A pluggable renderer cannot be loaded while a connection is active,
 * and if a pluggable decoder fails during a connection, the connection
 * will be terminated.
 *
 * Any strings which the filled struct contains are transferred to
 * the decoder, and should be deleted using VNCDecoderFactoryFree().
 *
 * \param decoderContext The decoder VNCDecoderContext that was passed
 * to VNCDecoderCreate().
 * \param pInfoStruct The struct which will be filled with the renderer's
 * information.
 * \param infoStructSize The length of the info struct, in bytes.
 * \return True if there is a pluggable renderer currently in use,
 * and false otherwise.
 */
typedef vnc_bool_t VNCCALL
VNCDecoderGetRendererInfo(VNCDecoderContext decoderContext,
                          VNCRendererIdentificationInfo *pInfoStruct,
                          size_t infoStructSize);

/**
 * \brief Called by the decoder to provide custom data to a pluggable
 * renderer.
 *
 * The pointer pData will be provided as-is to the pluggable renderer.
 * The data it points to should be in a format which the pluggable
 * renderer will understand.
 *
 * This method should only be called if the renderer which is loaded
 * is recognised by the decoder. This can be determined using the
 * VNCDecoderFactoryGetRendererInfo method.
 *
 * Calling this method will result in the pluggable renderer immediately
 * receiving a call to its VNCRendererDrawCustomData method. This method
 * will only return once VNCRendererDrawCustomData has returned.
 *
 * The rectangle and data provided must be valid for the duration of
 * this method call.
 *
 * \param decoderContext The VNCDecoderContext that was passed to
 * VNCDecoderCreate().
 * \param pRect The size and position of the rectangle to be drawn.
 * \param pData A pointer to some arbitrary data, which will be
 * provided as-is to the pluggable renderer.
 * \param dataSize The size of the data pointed to by pData.
 * \see VNCDecoderFactoryGetRendererInfo
 */
typedef VNCDecoderError VNCCALL
VNCDecoderDrawCustomData(VNCDecoderContext decoderContext,
                         const VNCRectangle *pRect,
                         const void* pData,
                         size_t dataSize);

/**
 * \brief APIs provided by the SDK for calling by the decoder.
 */
typedef struct {
  /**
   * Called by the decoder factory to write to the SDK's log.
   */
  VNCDecoderFactoryLog *vncDecoderFactoryLog;
  /**
   * Called by the decoder to allocate memory whose ownership is to be
   * transferred to the SDK.
   */
  VNCDecoderFactoryAlloc *vncDecoderFactoryAlloc;
  /**
   * Called by the decoder to allocate memory whose ownership is to be
   * transferred to the SDK.
   */
  VNCDecoderFactoryAllocString *vncDecoderFactoryAllocString;
  /**
   * Called by the decoder to free memory whose ownership has not been
   * transferred to the SDK.
   */
  VNCDecoderFactoryFree *vncDecoderFactoryFree;
  /**
   * Called by the decoder to write to the SDK's log.
   */
  VNCDecoderLog *vncDecoderLog;
  /**
   * Called by the decoder to check whether a particular features is licensed.
   */
  VNCDecoderLocalFeatureCheck *vncDecoderLocalFeatureCheck;
  /**
   * Called by the decoder to draw a rectangle to the SDK's framebuffer.
   */
  VNCDecoderDrawRectangle *vncDecoderDrawRectangle;
  /**
   * Called by the decoder to read data from an input stream.
   */
  VNCDecoderNonBlockingInputStreamRead *vncDecoderNonBlockingInputStreamRead;
  /**
   * Called by the decoder to access an input stream's internal buffer.
   */
  VNCDecoderNonBlockingInputStreamPeek *vncDecoderNonBlockingInputStreamPeek;
  /**
   * Called by the decoder to skip over data in an input stream.
   */
  VNCDecoderNonBlockingInputStreamSkip *vncDecoderNonBlockingInputStreamSkip;
  /**
   * Called by the decoder to determine whether or not a particular pluggable
   * renderer is active.
   */
  VNCDecoderGetRendererInfo *vncDecoderGetRendererInfo;
  /**
   * Called by the decoder to provide custom data to the loaded pluggable
   * renderer.
   */
  VNCDecoderDrawCustomData *vncDecoderDrawCustomData;

} VNCDecoderSupportingAPI;

/**
 * \brief The type of the entry point to be exported by a decoder DLL
 * or shared object.
 *
 * \param decoderFactoryContext Opaque data type to be provided to
 * applicable supporting APIs.
 * \param decoderViewerContext Opaque data type as provided by the
 * viewer application to the SDK when the decoder was registered or
 * loaded.
 * \param pDecoderInterface Structure to contain the addresses of the
 * decoders' implementation functions.  The implementation of
 * VNCDecoderFactoryInitialize() must fill in this structure before
 * returning.
 * \param decoderInterfaceSize The size of *pDecoderInterface.
 * \param pDecoderSupportingAPI Structure containing functions provided by the
 * SDK for calling by the decoder.
 * \param decoderSupportingAPISize The size of *pDecoderSupportingAPI.
 *
 * \return The new decoder object.  The SDK will call VNCDecoderTerminate() when
 * this object is no longer required. Returning NULL here indicates that an
 * error has occurred, and the decoder factory will not be initialised.
 */
typedef VNCDecoderFactory *VNCCALL VNCDecoderFactoryInitializeType(
    VNCDecoderFactoryContext decoderFactoryContext,
    VNCDecoderViewerContext decoderViewerContext,
    VNCDecoderInterface *pDecoderInterface,
    size_t decoderInterfaceSize,
    const VNCDecoderSupportingAPI *pDecoderSupportingAPI,
    size_t decoderSupportingAPISize);

#endif // VNCDECODER_H_013E722A_AB5D-4BA2_962E_83202E7C27AB

